/************************************************************************
 *
 * \file: iAPEANative2Session.cpp
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * <brief description>.
 * <detailed description>
 * \component: Baidu Carlife - Demo application
 *
 * \author: Ajay Kumar Sahoo / ajaykumar.sahoo@in.bosch.com
 *
 * \copyright (c) 2015 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 * \see <related items>
 *
 * \history
 *
 ***********************************************************************/
#include <adit_logging.h>

#include <stdlib.h>
#include <sys/unistd.h>  //read/close
#include <sys/syscall.h> //syscall
#include <sys/poll.h>    //POLLIN/POLLOUT
#include <sys/utsname.h> //utsname
#include <sys/prctl.h>
#include <sys/eventfd.h>
#include <errno.h>

#include <utils/iAP2EANativeSession.h>

#include <iap2_dlt_log.h>
#include <iap2_commands.h>


LOG_IMPORT_CONTEXT(tbdcl)


namespace adit { namespace bdcl {


iAP2EANativeSession::iAP2EANativeSession(std::string iap2devserial,EventDispatcher *EventDispatcher)
{
    // todo this context registration overrides the other iAP2 context registration settings, fix it
    IAP2REGISTERCTXTWITHDLT();

    mEventDispatcher = EventDispatcher;
    iAP2DevSerial = iap2devserial;
    p_iAP2Device = nullptr;
    memset(&iap2initparam,0,sizeof(iAP2InitParam_t));
    endPoll = false;

    iAP2AppThreadId = -1;
    pollThreadId    = -1;
    iAP2DevState    = iAP2ComError;

    mEventFd = -1;

    //Init semaphore for device ready wait
    (void)sem_init(&(iAP2DevReady), 0, 0);  //used in iAP2AppThread to wait until device ready
}


iAP2EANativeSession::~iAP2EANativeSession()
{
    sem_destroy(&iAP2DevReady);
    IAP2DEREGISTERCTXTWITHDLT();
}
int32_t iAP2EANativeSession::iAP2EASessionInit(void)
{
    int32_t res = 0;

    res = pthread_create(&iAP2AppThreadId, NULL, iAP2AppThread, this);
    if (0 == res) {
        LOG_INFO((tbdcl, "iAP2AppThread created"));
    } else {
        LOG_ERROR((tbdcl, "iAP2EANativeSession pthread_create failed: res=%d, errno=%s", res, strerror(errno)));
        res = -1;
    }

    return res;
}
int32_t iAP2EANativeSession::iAP2EASessionDeInit(void)
{
    int32_t res = 0;

    /* stop iAP2PollThread */
    endPoll = true;

    /* no specific value for event required */
    uint64_t event = 1;
    if (mEventFd >= 0) {
        /* trigger eventfd to wakeup iAP2PollThread */
        if (eventfd_write(mEventFd, event) != 0){
            LOG_ERROR((tbdcl, "iAP2EASessionDeInit() eventfd_write(fd=%d) failed", mEventFd));
        } else {
            LOGD_DEBUG((tbdcl, "iAP2EASessionDeInit() eventfd_write(fd=%d) to stop iAP2PollThread", mEventFd));
        }
    } else {
        LOG_ERROR((tbdcl, "iAP2EASessionDeInit() mEventFd=%d is invalid", mEventFd));
    }

    if (iAP2AppThreadId > 0) {
        LOG_INFO((tbdcl, "pthread_join iAP2AppThread"));
        int32_t err = pthread_join(iAP2AppThreadId, nullptr);
        if (0 == err) {
            LOG_INFO((tbdcl, "iAP2AppThread joined"));
            iAP2AppThreadId = -1;
        } else {
            LOG_ERROR((tbdcl, "pthread_join(iAP2AppThread) failed with error=%d, errno=%s", err, strerror(errno)));
            res = -1;
        }
    } else {
        LOG_INFO((tbdcl, "iAP2AppThread already joined"));
    }

    /* Close the eventfd after iAP2PollThread and iAP2AppThread has joined.
     * Otherwise, iAP2PollThread may could not read the stop event. */
    if (mEventFd >= 0) {
        close(mEventFd);
        mEventFd = -1;
    }
    return res;
}
void iAP2EANativeSession::iAP2ConfigureAccessoryBasics(iAP2AccessoryConfig_t* pConfig)
{
    pConfig->iAP2iOSintheCar        = false;
    pConfig->iAP2TransportType      = iAP2USBHOSTMODE; //to support EA native transport mode
    pConfig->iAP2AuthenticationType = iAP2AUTHI2C;

    //todo Implement pfcfg (below are strings. may get segfault)
    iAP2UtilCopyString((uint8_t*)IPOD_AUTH_DEV_NAME, &pConfig->iAP2AuthDevicename);
    iAP2UtilCopyString((uint8_t*)IPOD_AUTH_IOCTL_REG, &pConfig->iAP2AuthIoctlRegAddr);
    iAP2UtilCopyString((uint8_t*)IPOD_AUTH_GPIO_RESET, &pConfig->iAP2AuthGPIOReset);
    iAP2UtilCopyString((uint8_t*)IPOD_AUTH_GPIO_READY, &pConfig->iAP2AuthGPIOReady);


    pConfig->iAP2AuthShortWait      = IPOD_AUTH_DEV_COM_SHORT_WAIT;
    pConfig->iAP2AuthWait           = IPOD_AUTH_DEV_COM_WAIT;
    pConfig->iAP2AuthLongWait       = IPOD_AUTH_DEV_COM_LONG_WAIT;

    //Set accessory power configuration
    pConfig->iAP2AvailableCurrentForDevice                 = 1000;
    pConfig->iAP2DeviceBatteryShouldChargeIfPowerIsPresent = true;
    pConfig->iAP2MaximumcurrentDrawnFromAccessory          = true;
    pConfig->iAP2DeviceBatteryWillChargeIfPowerIsPresent   = true;
    pConfig->iAP2AccessoryPowerMode                        = true;
    pConfig->iAP2FileXferRcvAsStream                       = false;
    pConfig->iAP2EANativeTransport                         = true;

    pConfig->iAP2UsbOtgGPIOPower = (uint8_t*)BDCL_IAP2_ACC_CONFG_TRANS_USB_OTG_GPIO_POWER_AI;

    pConfig->iAP2EAPSupported                              = false;
    pConfig->iAP2FileXferSupported                         = false;

    pConfig->useConfigFS                                   = true;

}

int32_t  iAP2EANativeSession::iAP2ConfigureAccInfo(iAP2AccessoryInfo_t* pAccInfo, iAP2TransportType_t transportType)
{
    uint32_t i = 0;
    int32_t result = IAP2_OK;
    uint8_t AccessoryName[] = {BDCL_IAP2_ACC_INFO_NAME};
    uint8_t ModelIdentifier[] = {BDCL_IAP2_ACC_INFO_MODEL_IDENTIFIER};
    uint8_t Manufacturer[] = {BDCL_IAP2_ACC_INFO_MANUFACTURER};
    uint8_t SerialNumber[] = {BDCL_IAP2_ACC_INFO_SERIAL_NUM};
    uint8_t FirmwareVersion[] = {BDCL_IAP2_ACC_INFO_FW_VER};
    uint8_t HardwareVersion[] = {BDCL_IAP2_ACC_INFO_HW_VER};
    uint8_t idVendor[] = {BDCL_IAP2_ACC_INFO_VENDOR_ID};
    uint8_t idProduct[] = {BDCL_IAP2_ACC_INFO_PRODUCT_ID};
    uint8_t bcdDevice[] = {BDCL_IAP2_ACC_INFO_BCD_DEVICE};
    uint8_t initEndpoint[] = {BDCL_IAP2_ACC_INFO_INIT_ENDPOINT};
    uint8_t CurrentLanguage[] = {"de"};
    uint16_t SupportedLanguageCnt = 3;
    uint8_t SupportedLanguage[][3] = {"en", "de", "fr"};
    uint8_t iOSAppName[] = {BDCL_IAP2_APP_NAME};
    uint8_t iOSAppIdentifier = 1;
    uint32_t SupportediOSAppCount = 1;

    LOG_INFO((tbdcl,"%s(%p, type=%d)", __FUNCTION__, pAccInfo, transportType));

    memset(pAccInfo, 0, sizeof(iAP2AccessoryInfo_t));

    //Accessory name
    result = iAP2UtilCopyString(AccessoryName, &(pAccInfo->iAP2AccessoryName));
    if (IAP2_OK != result)
    {
        return result;
    }

    //Model identifier
    result = iAP2UtilCopyString(ModelIdentifier, &(pAccInfo->iAP2AccessoryModelIdentifier));
    if (IAP2_OK != result)
    {
        return result;
    }

    //Manufacturer
    result = iAP2UtilCopyString(Manufacturer, &(pAccInfo->iAP2AccessoryManufacturer));
    if (IAP2_OK != result)
    {
        return result;
    }

    //Serial number
    result = iAP2UtilCopyString(SerialNumber, &(pAccInfo->iAP2AccessorySerialNumber));
    if (IAP2_OK != result)
    {
        return result;
    }

    //Firmware version
    result = iAP2UtilCopyString(FirmwareVersion, &(pAccInfo->iAP2AccessoryFirmwareVersion));
    if (IAP2_OK != result)
    {
        return result;
    }

    //Hardware version
    result = iAP2UtilCopyString(HardwareVersion, &(pAccInfo->iAP2AccessoryHardwareVersion));
    if (IAP2_OK != result)
    {
        return result;
    }

    //Vendor ID
    result = iAP2UtilCopyString(idVendor, &(pAccInfo->iAP2AccessoryVendorId));
    if (IAP2_OK != result)
    {
        return result;
    }

    //Product ID
    result = iAP2UtilCopyString(idProduct, &(pAccInfo->iAP2AccessoryProductId));
    if (IAP2_OK != result)
    {
        return result;
    }

    //BCD device
    result = iAP2UtilCopyString(bcdDevice, &(pAccInfo->iAP2AccessoryBcdDevice));
    if (IAP2_OK != result)
    {
        return result;
    }

    //Init end point
    result = iAP2UtilCopyString(initEndpoint, &(pAccInfo->iAP2InitEndPoint));
    if (IAP2_OK != result)
    {
        return result;
    }

    //Messages received/sent
    if (transportType == iAP2USBHOSTMODE)
    {
        //Messages sent by accessory
        pAccInfo->iAP2CommandsUsedByApplication = (uint16_t*)calloc(1, sizeof(bdcl_demo_USBHostModeMsgSentByAcc));
        if(pAccInfo->iAP2CommandsUsedByApplication != NULL)
        {
            memcpy(pAccInfo->iAP2CommandsUsedByApplication, bdcl_demo_USBHostModeMsgSentByAcc,
                    sizeof(bdcl_demo_USBHostModeMsgSentByAcc));
            pAccInfo->iAP2CommandsUsedByApplication_length = sizeof(bdcl_demo_USBHostModeMsgSentByAcc);
        }
        else
        {
            LOG_INFO((tbdcl, "%s(%p) FATAL ERROR: Failed to allocate memory for messages sent by accessory",
                    __FUNCTION__, pAccInfo));
            return IAP2_ERR_NO_MEM;
        }

        //Messages received from device
        pAccInfo->iAP2CallbacksExpectedFromDevice = (uint16_t*)calloc(1, sizeof(bdcl_demo_USBHostModeMsgRecvFromDevice) );
        if (pAccInfo->iAP2CallbacksExpectedFromDevice != NULL)
        {
            memcpy(pAccInfo->iAP2CallbacksExpectedFromDevice, bdcl_demo_USBHostModeMsgRecvFromDevice, sizeof(bdcl_demo_USBHostModeMsgRecvFromDevice) );
            pAccInfo->iAP2CallbacksExpectedFromDevice_length = sizeof(bdcl_demo_USBHostModeMsgRecvFromDevice);
        }
        else
        {
            LOG_INFO((tbdcl,"%s(%p) FATAL ERROR: Failed to allocate memory for messages received by accessory",
                    __FUNCTION__, pAccInfo));
            return IAP2_ERR_NO_MEM;
        }
    }
    else
    {
        LOG_INFO((tbdcl, "%s(%p) FATAL ERROR: Wrong configuration: Transport type must be USB Host Mode!",
                __FUNCTION__, pAccInfo));

        return IAP2_BAD_PARAMETER;
    }

    //Current language
    result = iAP2UtilCopyString(CurrentLanguage, &(pAccInfo->iAP2CurrentLanguage));
    if (IAP2_OK != result)
    {
        LOG_INFO((tbdcl,"%s(%p) FATAL ERROR: Failed to allocate memory for current language",
                __FUNCTION__, pAccInfo));
        return result;
    }

    //Supported languages
    pAccInfo->iAP2SupportedLanguageCount = SupportedLanguageCnt;

    pAccInfo->iAP2SupportedLanguage = (uint8_t**)calloc(pAccInfo->iAP2SupportedLanguageCount, sizeof(uint8_t*));
    if (pAccInfo->iAP2SupportedLanguage != NULL)
    {
        for (i=0; i < pAccInfo->iAP2SupportedLanguageCount; i++)
        {
            result = iAP2UtilCopyString(&SupportedLanguage[i][0], &(pAccInfo->iAP2SupportedLanguage[i]));
            if (IAP2_OK != result)
            {
                LOG_INFO((tbdcl,"%s(%p) FATAL ERROR: Failed to allocate memory for supported language %d",
                        __FUNCTION__, pAccInfo, i));
                return result;
            }
        }
    }
    else
    {
        LOG_INFO((tbdcl,"%s(%p) FATAL ERROR: Failed to allocate memory for supported languages",
                __FUNCTION__, pAccInfo));
        return IAP2_ERR_NO_MEM;
    }

    pAccInfo->iAP2MaximumCurrentDrawnFromDevice = 0;

    //iOS in the Car is disabled
    pAccInfo->iAP2SupportsiOSintheCar = false;

    //Supported iOS applications
    pAccInfo->iAP2SupportediOSAppCount = SupportediOSAppCount;
    pAccInfo->iAP2iOSAppInfo = (iAP2iOSAppInfo_t*)calloc(pAccInfo->iAP2SupportediOSAppCount, sizeof(iAP2iOSAppInfo_t));
    if (pAccInfo->iAP2iOSAppInfo != NULL)
    {
        for (i=0; i < pAccInfo->iAP2SupportediOSAppCount; i++)
        {
            pAccInfo->iAP2iOSAppInfo[i].iAP2EANativeTransport = true; //must be true otherwise identification gets rejected
            pAccInfo->iAP2iOSAppInfo[i].iAP2iOSAppIdentifier = iOSAppIdentifier;
            pAccInfo->iAP2iOSAppInfo[i].iAP2EAPMatchAction = IAP2_USER_APP_MATCH; //No prompt but user can manually search for application in Settings -> General -> About
            result = iAP2UtilCopyString(iOSAppName, &(pAccInfo->iAP2iOSAppInfo[i].iAP2iOSAppName));
            if (IAP2_OK != result)
            {
                LOG_INFO((tbdcl,"%s(%p) FATAL ERROR: Failed to allocate memory for iOS app name",
                        __FUNCTION__, pAccInfo));
                return result;
            }
        }
    }
    else
    {
        LOG_INFO((tbdcl,"%s(%p) FATAL ERROR: Failed to allocate memory for iOS app info",
                __FUNCTION__, pAccInfo));
        return IAP2_ERR_NO_MEM;
    }

//    //App bundle seed identifier
//    result = mspin_demo_iap2_copyString(iOSAppBundleSeedId, &(pAccInfo->iAP2PreferredAppBundleSeedIdentifier));
//    if (IAP2_OK != result)
//    {
//        return result;
//    }

    return IAP2_OK;

}
void iAP2EANativeSession::iAP2ConfigureSessionCallbacks(iAP2SessionCallbacks_t* pCallbacks)
{
     pCallbacks->iAP2AuthenticationFailed_cb                   = &iAP2AuthenticationFailedCB;
     pCallbacks->iAP2AuthenticationSucceeded_cb                = &iAP2AuthenticationSucceededCB;
     pCallbacks->iAP2RequestAuthenticationCertificate_cb       = NULL;
     pCallbacks->iAP2RequestAuthenticationChallengeResponse_cb = NULL;
     pCallbacks->iAP2StartIdentification_cb                    = NULL;
     pCallbacks->iAP2IdentificationAccepted_cb                 = &iAP2IdentificationAcceptedCB;
     pCallbacks->iAP2IdentificationRejected_cb                 = &iAP2IdentificationRejectedCB;
     pCallbacks->iAP2AssistiveTouchInformation_cb              = NULL;
     pCallbacks->iAP2BluetoothConnectionUpdate_cb              = NULL;
     pCallbacks->iAP2DeviceAuthenticationCertificate_cb        = NULL;
     pCallbacks->iAP2DeviceAuthenticationResponse_cb           = NULL;
     pCallbacks->iAP2DeviceInformationUpdate_cb                = NULL;
     pCallbacks->iAP2DeviceLanguageUpdate_cb                   = NULL;
     pCallbacks->iAP2StartExternalAccessoryProtocolSession_cb  = NULL; //EAP start session callbacks will not be needed
     pCallbacks->iAP2StopExternalAccessoryProtocolSession_cb   = NULL; //EAP stop session callbacks will not be needed
     pCallbacks->iAP2DeviceHIDReport_cb                        = NULL;
     pCallbacks->iAP2StartLocationInformation_cb               = NULL;
     pCallbacks->iAP2StopLocationInformation_cb                = NULL;
     pCallbacks->iAP2MediaLibraryInformation_cb                = NULL;
     pCallbacks->iAP2MediaLibraryUpdate_cb                     = NULL;
     pCallbacks->iAP2NowPlayingUpdateParameter_cb              = NULL;
     pCallbacks->iAP2PowerUpdate_cb                            = &iAP2PowerUpdateCB;
     pCallbacks->iAP2TelephonyCallStateInformation_cb          = NULL;
     pCallbacks->iAP2TelephonyUpdate_cb                        = NULL;
     pCallbacks->iAP2USBDeviceModeAudioInformation_cb          = NULL;
     pCallbacks->iAP2StartVehicleStatusUpdates_cb              = NULL;
     pCallbacks->iAP2StopVehicleStatusUpdates_cb               = NULL;
     pCallbacks->iAP2VoiceOverCursorUpdate_cb                  = NULL;
     pCallbacks->iAP2VoiceOverUpdate_cb                        = NULL;
     pCallbacks->iAP2WiFiInformation_cb                        = NULL;

}
void iAP2EANativeSession::iAP2ConfigureStackCallbacks(iAP2StackCallbacks_t* pStackCallbacks)
{
        pStackCallbacks->p_iAP2DeviceState_cb = &iAP2UpdateDeviceStateCB;
}

void iAP2EANativeSession::iAP2ConfigureEANativeTransportCallbacks(iAP2EANativeTransportCallbacks_t* pEANativeTransportCallbacks)
{
       pEANativeTransportCallbacks->p_iAP2StartEANativeTransport_cb = iAP2StartEANativeTransportCB;
       pEANativeTransportCallbacks->p_iAP2StopEANativeTransport_cb = iAP2StopEANativeTransportCB;

}

int32_t iAP2EANativeSession::iAP2ConfigureAccessory()
{
    int32_t rc = IAP2_OK;

    iap2initparam.iAP2ContextCallback = this;

    //set device serial number
    strncpy((char*)iap2initparam.iAP2DeviceId, iAP2DevSerial.c_str(), strlen(iAP2DevSerial.c_str())+1);

    //Basic configuration
    iap2initparam.p_iAP2AccessoryConfig = (iAP2AccessoryConfig_t*)calloc(1, sizeof(iAP2AccessoryConfig_t));
    if (NULL == iap2initparam.p_iAP2AccessoryConfig)
    {
        LOG_INFO((tbdcl,
                "%s FATAL ERROR: Failed to allocate buffer for p_iAP2AccessoryConfig",
                __FUNCTION__));
        return IAP2_ERR_NO_MEM;
    }

    iAP2ConfigureAccessoryBasics(iap2initparam.p_iAP2AccessoryConfig);

    //Configure accessory info
    iap2initparam.p_iAP2AccessoryInfo = (iAP2AccessoryInfo_t*)calloc(1, sizeof(iAP2AccessoryInfo_t));
    if (NULL ==iap2initparam.p_iAP2AccessoryInfo)
    {
        LOG_INFO((tbdcl,
                "%s FATAL ERROR: Failed to allocate buffer for p_iAP2AccessoryInfo",
                __FUNCTION__));
        return IAP2_ERR_NO_MEM;
    }

    iAP2ConfigureAccInfo(iap2initparam.p_iAP2AccessoryInfo,
            iap2initparam.p_iAP2AccessoryConfig->iAP2TransportType);

    //Configure iAP2 callbacks
    iap2initparam.p_iAP2CSCallbacks = (iAP2SessionCallbacks_t*)calloc(1, sizeof(iAP2SessionCallbacks_t));
    if (NULL == iap2initparam.p_iAP2CSCallbacks)
    {
        LOG_INFO((tbdcl,
                "%s FATAL ERROR: Failed to allocate buffer for p_iAP2CSCallbacks",
                __FUNCTION__));

        return IAP2_ERR_NO_MEM;
    }
    iAP2ConfigureSessionCallbacks(iap2initparam.p_iAP2CSCallbacks);

    //Configure transfer callbacks is not required because iAP2FileXferSupported should be set to false)
    if (iap2initparam.p_iAP2AccessoryConfig->iAP2FileXferSupported)
    {
        LOG_INFO((tbdcl,
                "%s  ERROR: iAP2FileXferSupported should be disabled but is enabled!",
                __FUNCTION__));
    }

    //Configure stack callbacks
    iap2initparam.p_iAP2StackCallbacks = (iAP2StackCallbacks_t*)calloc(1, sizeof(iAP2StackCallbacks_t));
    if (NULL == iap2initparam.p_iAP2StackCallbacks)
    {
        LOG_INFO((tbdcl,
                "%s FATAL ERROR: Failed to allocate memory for iAP2StackCallbacks",
                __FUNCTION__));

        return IAP2_ERR_NO_MEM;
    }
    iAP2ConfigureStackCallbacks(iap2initparam.p_iAP2StackCallbacks);

    //Configure EA native transport callbacks (do not use EAP callbacks)
    iap2initparam.p_iAP2EANativeTransportCallbacks = (iAP2EANativeTransportCallbacks_t*)calloc(1, sizeof(iAP2EANativeTransportCallbacks_t));
    if (NULL == iap2initparam.p_iAP2EANativeTransportCallbacks)
    {
        LOG_INFO((tbdcl,
                "%s FATAL ERROR: Failed to allocate memory for iAP2EANativeTransportCallbacks",
                __FUNCTION__));

        return IAP2_ERR_NO_MEM ;
    }
    iAP2ConfigureEANativeTransportCallbacks(iap2initparam.p_iAP2EANativeTransportCallbacks);

    return rc;

}

int32_t iAP2EANativeSession::iAP2UtilCopyString(uint8_t* pString, uint8_t** ppDestination)
{
    int32_t StringLength = strnlen((const char*)pString, STRING_MAX);
    *ppDestination = (uint8_t*)calloc(1, (StringLength + IAP2_NULL_CHAR_LEN));
    if (*ppDestination != NULL)
    {
        memcpy(*ppDestination, pString, (StringLength + IAP2_NULL_CHAR_LEN) );
        (*ppDestination)[StringLength] = '\0';

        return IAP2_OK;
    }
    else
    {
        LOG_INFO((tbdcl, "%s('%s') FATAL ERROR: Could not allocate memory for string", __func__, pString));
        return IAP2_ERR_NO_MEM;
    }
}

int32_t iAP2EANativeSession::iAP2UpdateDeviceStateCB(iAP2Device_t* iap2Device, iAP2DeviceState_t state, void* context)
{
    int32_t rc = IAP2_OK;

    auto eaObj = static_cast<iAP2EANativeSession*>(context);

    switch (state)
    {
        case iAP2NotConnected:
        {
            eaObj->iAP2DevState = iAP2NotConnected;
            LOG_INFO((tbdcl,
                    "%s(device=%p) new state = iAP2NotConnected",
                    __FUNCTION__, iap2Device));

            break;
        }
        case iAP2LinkConnected:
        {
            eaObj->iAP2DevState = iAP2LinkConnected;
            LOG_INFO((tbdcl,
                    "%s(device=%p) new state = iAP2LinkConnected",
                    __FUNCTION__, iap2Device));
            break;
        }
        case iAP2AuthenticationPassed:
        {
            eaObj->iAP2DevState = iAP2AuthenticationPassed;
            LOG_INFO((tbdcl,
                    "%s(device=%p) new state = iAP2AuthenticationPassed",
                    __FUNCTION__, iap2Device));
            break;
        }
        case iAP2IdentificationPassed:
        {
            eaObj->iAP2DevState = iAP2IdentificationPassed;
            LOG_INFO((tbdcl,
                    "%s(device=%p) new state = iAP2IdentificationPassed",
                    __FUNCTION__, iap2Device));
            break;
        }
        case iAP2DeviceReady:
        {
            LOG_INFO((tbdcl,
                    "%s(device=%p) new state = iAP2DeviceReady",
                    __FUNCTION__, iap2Device));
            eaObj->iAP2DevState = iAP2DeviceReady;

            //release iAP2AppThread to do other commands
            sem_post(&(eaObj->iAP2DevReady));
            break;
        }
        case iAP2LinkiAP1DeviceDetected:
        {
            eaObj->iAP2DevState = iAP2LinkiAP1DeviceDetected;

            LOG_INFO((tbdcl,
                    "%s(device=%p) new state = iAP2LinkiAP1DeviceDetected",
                    __FUNCTION__, iap2Device));
            break;
        }
        case iAP2TransportConnected:
        {
            eaObj->iAP2DevState = iAP2TransportConnected;

            LOG_INFO((tbdcl,
                    "%s() new device=%p state = iAP2TransportConnected",
                    __FUNCTION__, iap2Device));
            break;
        }
        default:
        {
            eaObj->iAP2DevState = iAP2ComError;

            LOG_INFO((tbdcl,
                    "%s(device=%p) new state = unknown (%d)",
                    __FUNCTION__, iap2Device, state));
            rc = IAP2_CTL_ERROR;
            break;
        }
    }

    return rc;
}

int32_t iAP2EANativeSession::iAP2StartEANativeTransportCB(iAP2Device_t* iap2Device, uint8_t iAP2iOSAppIdentifier, uint8_t sinkEndpoint, uint8_t sourceEndpoint, void* context)
{
    std::string eaWStr;
    std::string eaRStr;
    std::string ffsStr = "/dev/ffs/ep";

    LOG_INFO((tbdcl,"%s(device=%p, iOSAppId=%d, SinkEndpoint=%d, SourceEndpoint=%d, ctx=%p)",
            __FUNCTION__, iap2Device, iAP2iOSAppIdentifier,sinkEndpoint, sourceEndpoint, context));

    auto eaObj = static_cast<iAP2EANativeSession*>(context);

    eaWStr  = ffsStr + std::to_string(sinkEndpoint);
    eaRStr  = ffsStr + std::to_string(sourceEndpoint);

    //create a new event item and deligate to Server
    std::shared_ptr<EventItem> event(new EventItem(nullptr, EventType::EANSTART, eaRStr,eaWStr));
    eaObj->mEventDispatcher->queueEvent(event);

    return IAP2_OK;
}

int32_t iAP2EANativeSession::iAP2StopEANativeTransportCB(iAP2Device_t* iap2Device, uint8_t iAP2iOSAppIdentifier, uint8_t sinkEndpoint, uint8_t sourceEndpoint, void* context)
{
    std::string eaWStr;
    std::string eaRStr;
    std::string ffsStr = "/dev/ffs/ep";

    LOG_INFO((tbdcl, "%s(device=%p, iOSAppId=%d, sinkEp?%d, sourcEp=%d, ctx=%p)",
            __FUNCTION__, iap2Device, iAP2iOSAppIdentifier, sinkEndpoint, sourceEndpoint, context));

    auto eaObj = static_cast<iAP2EANativeSession*>(context);

    eaWStr  = ffsStr + std::to_string(sinkEndpoint);
    eaRStr  = ffsStr + std::to_string(sourceEndpoint);

    //create a new event item and deligate to Serverr
    std::shared_ptr<EventItem> event(new EventItem(nullptr, EventType::EANSTOP, eaRStr,eaWStr));
    eaObj->mEventDispatcher->queueEvent(event);

    return IAP2_OK;
}

int32_t iAP2EANativeSession::iAP2AuthenticationFailedCB(iAP2Device_t* iap2Device, iAP2AuthenticationFailedParameter* authParameter, void* context)
{
    (void)authParameter;
    (void)iap2Device;
// todo Get rid of #if 0
#if 0
    LOG_INFO((tbdcl, "%s(...,ctx=%p)", __FUNCTION__, context));
#endif
    return IAP2_CTL_ERROR;
}

int32_t iAP2EANativeSession::iAP2AuthenticationSucceededCB(iAP2Device_t* iap2Device, iAP2AuthenticationSucceededParameter* authParameter, void* context)
{
    (void)authParameter;
    (void)iap2Device;
// todo Get rid of #if 0
#if 0
    LOG_INFO((tbdcl,"%s(..., ctx=%p)", __FUNCTION__, context));
#endif
    return IAP2_OK;
}

int32_t iAP2EANativeSession::iAP2IdentificationAcceptedCB(iAP2Device_t* iap2Device, iAP2IdentificationAcceptedParameter* idParameter, void* context)
{
    (void)idParameter;
    (void)context;
    (void)iap2Device;
// todo Get rid of #if 0
#if 0
    LOG_INFO((tbdcl,"%s(..., ctx=%p)", __FUNCTION__, context));
#endif
    return IAP2_OK;
}

int32_t iAP2EANativeSession::iAP2IdentificationRejectedCB(iAP2Device_t* iap2Device, iAP2IdentificationRejectedParameter* idParameter, void* context)
{
    (void)iap2Device;

    LOG_INFO((tbdcl,"%s(..., ctx=%p) ERROR", __FUNCTION__, context));
// todo Get rid of #if 0
#if 0
    mspin_demo_iap2_parseIdRejectedMsg(idParameter);
#endif

    return IAP2_CTL_ERROR;
}

int32_t iAP2EANativeSession::iAP2PowerUpdateCB(iAP2Device_t* iap2Device, iAP2PowerUpdateParameter* powerupdateParameter, void* context)
{
    int32_t rc = IAP2_CTL_ERROR;

    (void)iap2Device;
    (void)powerupdateParameter;
// todo Get rid of #if 0
#if 0
    LOG_INFO((tbdcl,"%s(..., ctx=%p)", __FUNCTION__, context));

    if (powerupdateParameter->iAP2MaximumCurrentDrawnFromAccessory_count != 0)
    {
        rc = IAP2_OK;
    }
    if (powerupdateParameter->iAP2AccessoryPowerMode_count != 0)
    {
        rc = IAP2_OK;
    }
    if (powerupdateParameter->iAP2DeviceBatteryWillChargeIfPowerIsPresent_count != 0)
    {
        rc = IAP2_OK;
    }
#endif

    return rc;
}
void iAP2EANativeSession::iAP2ParseIdRejectedMsg(iAP2IdentificationRejectedParameter* idParameter)
{
    uint16_t i = 0;

    if (idParameter)
    {
        //Firmware version
        if (idParameter->iAP2AccessoryFirmwareVersion_count> 0)
        {
            LOG_INFO((tbdcl,
                    "%s() ERROR: firmware version count is %d",
                    __FUNCTION__, idParameter->iAP2AccessoryFirmwareVersion_count));

            for (i = 0; i < idParameter->iAP2AccessoryFirmwareVersion_count; i++)
            {
                LOG_INFO((tbdcl,
                        "%s() ERROR: %d firmware version is %s",
                        __FUNCTION__,  i,
                        (0 == idParameter->iAP2AccessoryFirmwareVersion[i]) ? "false" : "true"));
            }
        }

        //Hardware version
        if (idParameter->iAP2AccessoryHardwareVersion_count > 0)
        {
            LOG_INFO((tbdcl,
                    "%s() ERROR: hardware version count is %d",
                    __FUNCTION__, idParameter->iAP2AccessoryHardwareVersion_count));

            for (i = 0; i < idParameter->iAP2AccessoryHardwareVersion_count; i++)
            {
                LOG_INFO((tbdcl,
                        "%s() ERROR: %d firmware version is %s",
                        __FUNCTION__, i,
                        (0 == idParameter->iAP2AccessoryHardwareVersion[i]) ? "false" : "true"));
            }
        }

        //Manufacturer
        if (idParameter->iAP2AccessoryManufacturer_count > 0)
        {
            LOG_INFO((tbdcl,
                    "%s() ERROR: manufacturer count is %d",
                    __FUNCTION__, idParameter->iAP2AccessoryManufacturer_count));
        }

        //Model identifier
        if (idParameter->iAP2AccessoryModelIdentifier_count > 0)
        {
            LOG_INFO((tbdcl,
                    "%s() ERROR: model identifier count is %d",
                    __FUNCTION__, idParameter->iAP2AccessoryModelIdentifier_count));
        }

        //Accessory name
        if (idParameter->iAP2AccessoryHardwareVersion_count > 0)
        {
            LOG_INFO((tbdcl,
                    "%s() ERROR: accessory hardware version count is %d",
                    __FUNCTION__, idParameter->iAP2AccessoryName_count));

            for (i = 0; i < idParameter->iAP2AccessoryHardwareVersion_count; i++)
            {
                LOG_INFO((tbdcl,
                        "%s() ERROR: %d accessory name is %s",
                        __FUNCTION__, i,
                        (0 == idParameter->iAP2AccessoryName[i]) ? "false" : "true"));
            }
        }

        //Accessory serial number
        if (idParameter->iAP2AccessorySerialNumber_count > 0)
        {
            LOG_INFO((tbdcl,
                    "%s() ERROR: accessory serial number count is %d",
                    __FUNCTION__, idParameter->iAP2AccessorySerialNumber_count));
        }

        //Bluetooth transport component
        if (idParameter->iAP2BluetoothTransportComponent_count > 0)
        {
            LOG_INFO((tbdcl,
                    "%s() ERROR: bluetooth transport component count is %d",
                    __FUNCTION__, idParameter->iAP2BluetoothTransportComponent_count));
        }

        //Current language
        if (idParameter->iAP2CurrentLanguage_count > 0)
        {
            LOG_INFO((tbdcl,
                    "%s() ERROR: current language count is %d",
                    __FUNCTION__, idParameter->iAP2CurrentLanguage_count));
        }

        //HID component
        if (idParameter->iAP2iAP2HIDComponent_count > 0)
        {
            LOG_INFO((tbdcl,
                    "%s() ERROR: HID component count is %d",
                    __FUNCTION__, idParameter->iAP2iAP2HIDComponent_count));
        }

        //Location information component
        if (idParameter->iAP2LocationInformationComponent_count > 0)
        {
            LOG_INFO((tbdcl,
                    "%s() ERROR: location information component count is %d",
                    __FUNCTION__, idParameter->iAP2LocationInformationComponent_count));
        }

        //Location information component
        if (idParameter->iAP2MaximumCurrentDrawnFromDevice_count > 0)
        {
            LOG_INFO((tbdcl,
                    "%s() ERROR: maximum current drawn from device count is %d",
                    __FUNCTION__, idParameter->iAP2MaximumCurrentDrawnFromDevice_count));
        }

        //Message received from device
        if (idParameter->iAP2MessagesRecievedfromDevice_count > 0)
        {
            LOG_INFO((tbdcl,
                    "%s() ERROR: messages received from device count is %d",
                    __FUNCTION__, idParameter->iAP2MessagesRecievedfromDevice_count));
        }

        //Messages sent by accessory
        if (idParameter->iAP2MessagesSentByAccessory_count > 0)
        {
            LOG_INFO((tbdcl,
                    "%s() ERROR: messages sent by accessory count is %d",
                    __FUNCTION__, idParameter->iAP2MessagesSentByAccessory_count));
        }

        //Power source type
        if (idParameter->iAP2PowerSourceType_count > 0)
        {
            LOG_INFO((tbdcl,
                    "%s() ERROR: power source type count is %d",
                    __FUNCTION__, idParameter->iAP2PowerSourceType_count));
        }

        //Preferred app bundle seed identifier
        if (idParameter->iAP2PreferredAppBundleSeedIdentifier_count > 0)
        {
            LOG_INFO((tbdcl,
                    "%s() ERROR: preferred app bundle seed identifier count is %d",
                    __FUNCTION__, idParameter->iAP2PreferredAppBundleSeedIdentifier_count));
        }

        //Serial transport component
        if (idParameter->iAP2SerialTransportComponent_count > 0)
        {
            LOG_INFO((tbdcl,
                    "%s() ERROR: serial transport component count is %d",
                    __FUNCTION__, idParameter->iAP2SerialTransportComponent_count));
        }

        //Supported External Accessory protocol
        if (idParameter->iAP2SupportedExternalAccessoryProtocol_count > 0)
        {
            LOG_INFO((tbdcl,
                    "%s() ERROR: Supported External Accessory protocol count is %d",
                    __FUNCTION__, idParameter->iAP2SupportedExternalAccessoryProtocol_count));
        }

        //Supported language
        if (idParameter->iAP2SupportedLanguage_count > 0)
        {
            LOG_INFO((tbdcl,
                    "%s() ERROR: Supported language count is %d",
                    __FUNCTION__, idParameter->iAP2SupportedLanguage_count));
        }

        //USB device transport component
        if (idParameter->iAP2USBDeviceTransportComponent_count)
        {
            LOG_INFO((tbdcl,
                    "%s() ERROR: USB device transport component count is %d",
                    __FUNCTION__, idParameter->iAP2USBDeviceTransportComponent_count));
        }

        //USB host HID component
        if (idParameter->iAP2USBHostHIDComponent_count > 0)
        {
            LOG_INFO((tbdcl,
                    "%s() ERROR: USB host HID component count is %d",
                    __FUNCTION__, idParameter->iAP2USBHostHIDComponent_count));
        }

        //USB host transport component
        if (idParameter->iAP2USBHostTransportComponent_count > 0)
        {
            LOG_INFO((tbdcl,
                    "%s() ERROR: USB host transport component count is %d",
                    __FUNCTION__, idParameter->iAP2USBHostTransportComponent_count));
        }

        //Vehicle information component
        if (idParameter->iAP2VehicleInformationComponent_count > 0)
        {
            LOG_INFO((tbdcl,
                    "%s() ERROR: Vehicle information component is %d",
                    __FUNCTION__, idParameter->iAP2VehicleInformationComponent_count));
        }

        //Vehicle status component
        if (idParameter->iAP2VehicleStatusComponent_count > 0)
        {
            LOG_INFO((tbdcl,
                    "%s() ERROR: Vehicle status component is %d",
                    __FUNCTION__, idParameter->iAP2VehicleStatusComponent_count));
        }
    }
    else
    {
        LOG_INFO((tbdcl,"%s() ERRROR: idParameter is NULL", __FUNCTION__));
    }
}
void iAP2EANativeSession::iAP2FreeAccInfo(iAP2AccessoryInfo_t *pAccInfo)
{
    uint16_t i = 0;

    if (!pAccInfo )
    {
        return;
    }

    if (pAccInfo->iAP2AccessoryName != NULL)
    {
        free(pAccInfo->iAP2AccessoryName);
        pAccInfo->iAP2AccessoryName = NULL;
    }

    if (pAccInfo->iAP2AccessoryModelIdentifier != NULL)
    {
        free(pAccInfo->iAP2AccessoryModelIdentifier);
        pAccInfo->iAP2AccessoryModelIdentifier = NULL;
    }

    if (pAccInfo->iAP2AccessoryManufacturer != NULL)
    {
        free(pAccInfo->iAP2AccessoryManufacturer);
        pAccInfo->iAP2AccessoryManufacturer = NULL;
    }

    if (pAccInfo->iAP2AccessorySerialNumber != NULL)
    {
        free(pAccInfo->iAP2AccessorySerialNumber);
        pAccInfo->iAP2AccessorySerialNumber = NULL;
    }

    if (pAccInfo->iAP2AccessoryFirmwareVersion != NULL)
    {
        free(pAccInfo->iAP2AccessoryFirmwareVersion);
        pAccInfo->iAP2AccessoryFirmwareVersion = NULL;
    }

    if (pAccInfo->iAP2AccessoryHardwareVersion != NULL)
    {
        free(pAccInfo->iAP2AccessoryHardwareVersion);
        pAccInfo->iAP2AccessoryHardwareVersion = NULL;
    }

    if (pAccInfo->iAP2CommandsUsedByApplication != NULL)
    {
        free(pAccInfo->iAP2CommandsUsedByApplication);
        pAccInfo->iAP2CommandsUsedByApplication = NULL;
    }

    if (pAccInfo->iAP2CallbacksExpectedFromDevice != NULL)
    {
        free(pAccInfo->iAP2CallbacksExpectedFromDevice);
        pAccInfo->iAP2CallbacksExpectedFromDevice = NULL;
    }

    if (pAccInfo->iAP2PreferredAppBundleSeedIdentifier != NULL)
    {
        free(pAccInfo->iAP2PreferredAppBundleSeedIdentifier);
        pAccInfo->iAP2PreferredAppBundleSeedIdentifier = NULL;
    }

    if (pAccInfo->iAP2CurrentLanguage != NULL)
    {
        free(pAccInfo->iAP2CurrentLanguage);
        pAccInfo->iAP2CurrentLanguage = NULL;
    }

    if (pAccInfo->iAP2SupportedLanguage != NULL)
    {
        for (i=0; i<pAccInfo->iAP2SupportedLanguageCount; i++)
        {
            if (pAccInfo->iAP2SupportedLanguage[i] != NULL)
            {
                free(pAccInfo->iAP2SupportedLanguage[i]);
                pAccInfo->iAP2SupportedLanguage[i] = NULL;
            }
        }
        free(pAccInfo->iAP2SupportedLanguage);
        pAccInfo->iAP2SupportedLanguage = NULL;
    }

    if (pAccInfo->iAP2iOSAppInfo != NULL)
    {
        for (i=0; i<pAccInfo->iAP2SupportediOSAppCount; i++)
        {

                pAccInfo->iAP2iOSAppInfo[i].iAP2iOSAppIdentifier = 0;
                if (pAccInfo->iAP2iOSAppInfo[i].iAP2iOSAppName != NULL)
                {
                    free(pAccInfo->iAP2iOSAppInfo[i].iAP2iOSAppName);
                    pAccInfo->iAP2iOSAppInfo[i].iAP2iOSAppName = NULL;
                }
        }
        free(pAccInfo->iAP2iOSAppInfo);
        pAccInfo->iAP2iOSAppInfo = NULL;
        pAccInfo->iAP2SupportediOSAppCount = 0;
    }

    if (pAccInfo->iAP2USBDeviceSupportedAudioSampleRate != NULL)
    {
        free(pAccInfo->iAP2USBDeviceSupportedAudioSampleRate);
        pAccInfo->iAP2USBDeviceSupportedAudioSampleRate = NULL;
    }

    if (pAccInfo->iAP2BluetoothTransportMAC != NULL)
    {
        free(pAccInfo->iAP2BluetoothTransportMAC);
        pAccInfo->iAP2BluetoothTransportMAC = NULL;
    }

    //ToDo: free other parameters
}

void iAP2EANativeSession::iAP2FreeAccConfig(iAP2AccessoryConfig_t* p_iAP2AccessoryConfig)
{
    if (p_iAP2AccessoryConfig->iAP2AuthDevicename != NULL)
    {
        free(p_iAP2AccessoryConfig->iAP2AuthDevicename);
        p_iAP2AccessoryConfig->iAP2AuthDevicename = NULL;
    }
    if (p_iAP2AccessoryConfig->iAP2AuthIoctlRegAddr != NULL)
    {
        free(p_iAP2AccessoryConfig->iAP2AuthIoctlRegAddr);
        p_iAP2AccessoryConfig->iAP2AuthIoctlRegAddr = NULL;
    }
    if (p_iAP2AccessoryConfig->iAP2AuthGPIOReset != NULL)
    {
        free(p_iAP2AccessoryConfig->iAP2AuthGPIOReset);
        p_iAP2AccessoryConfig->iAP2AuthGPIOReset = NULL;
    }
    if (p_iAP2AccessoryConfig->iAP2AuthGPIOReady != NULL)
    {
        free(p_iAP2AccessoryConfig->iAP2AuthGPIOReady);
        p_iAP2AccessoryConfig->iAP2AuthGPIOReady = NULL;
    }
}


void iAP2EANativeSession::iAP2DeConfigureAccessory()
{
    /*TODO: Add more de-initialization functions to free the dynamic parts of the structures*/

    if (NULL != iap2initparam.p_iAP2AccessoryInfo)
    {
        iAP2FreeAccInfo(iap2initparam.p_iAP2AccessoryInfo);
        free(iap2initparam.p_iAP2AccessoryInfo);
        iap2initparam.p_iAP2AccessoryInfo = NULL;
    }

    if(NULL != iap2initparam.p_iAP2AccessoryConfig)
    {
        iAP2FreeAccConfig(iap2initparam.p_iAP2AccessoryConfig);
        free(iap2initparam.p_iAP2AccessoryConfig);
        iap2initparam.p_iAP2AccessoryConfig = NULL;
    }
    if (NULL != iap2initparam.p_iAP2AccessoryConfig)
    {

        free(iap2initparam.p_iAP2AccessoryConfig);
        iap2initparam.p_iAP2AccessoryConfig = NULL;
    }

    if (NULL != iap2initparam.p_iAP2CSCallbacks)
    {
        free(iap2initparam.p_iAP2CSCallbacks);
        iap2initparam.p_iAP2CSCallbacks = NULL;
    }

    if (NULL != iap2initparam.p_iAP2EAPSessionCallbacks)
    {
        free(iap2initparam.p_iAP2EAPSessionCallbacks);
        iap2initparam.p_iAP2EAPSessionCallbacks = NULL;
    }

    if (NULL != iap2initparam.p_iAP2StackCallbacks)
    {
        free(iap2initparam.p_iAP2StackCallbacks);
        iap2initparam.p_iAP2StackCallbacks = NULL;
    }

    if (NULL != iap2initparam.p_iAP2EANativeTransportCallbacks)
    {
        free(iap2initparam.p_iAP2EANativeTransportCallbacks);
        iap2initparam.p_iAP2EANativeTransportCallbacks = NULL;
    }
}


int32_t iAP2EANativeSession::addFDToPollFDs(iAP2PollFDs_t* getPollFDs, int32_t numFDs, int32_t fdToAdd, int16_t eventToAdd)
{
    int32_t rc = IAP2_OK;
    int32_t i = 0;

    if (getPollFDs == NULL)
    {
        rc = IAP2_CTL_ERROR;
    }
    else
    {
        i = numFDs;

        getPollFDs[i].fd = fdToAdd;
        getPollFDs[i].event = eventToAdd;

        i++;
        rc = i;
    }

    return rc;
}
int32_t iAP2EANativeSession::addFDsToFDset(iAP2PollFDs_t* getPollFDs, int32_t numFDs, int32_t* maxfd, fd_set* to_readfds, fd_set* to_writefds)
{
    int32_t rc = 0;
    int32_t i = 0;

    if((getPollFDs == NULL) || (to_readfds == NULL)
        || (to_writefds == NULL) || (maxfd == NULL))
    {
        rc = -1;
    }
    else
    {
        /* adds the file descriptors to the fd_set */
        for(i = 0; i < numFDs; i++)
        {
            /* find highest-numbered file descriptor */
            if(getPollFDs[i].fd > *maxfd)
            {
                *maxfd = getPollFDs[i].fd;
            }

            if(getPollFDs[i].event == POLLIN)
            {
                /* FD_SET() adds the file descriptor to the fd_set */
                FD_SET(getPollFDs[i].fd, to_readfds);
                LOGD_DEBUG((tbdcl, "%s()  fd=%d is set to read fds, event %d",
                        __FUNCTION__, getPollFDs[i].fd, getPollFDs[i].event));
            }
            else if(getPollFDs[i].event == POLLOUT)
            {
                /* FD_SET() adds the file descriptor to the fd_set */
                FD_SET(getPollFDs[i].fd, to_writefds);
                LOGD_DEBUG((tbdcl, "%s()  fd=%d is set to write fds, event %d",
                        __FUNCTION__, getPollFDs[i].fd, getPollFDs[i].event));
            }
            else
            {
                LOG_WARN((tbdcl, "%s()  fd=%d is used for unknown event %d",
                        __FUNCTION__, getPollFDs[i].fd, getPollFDs[i].event));
            }
        }
        rc = 0;
    }

    return rc;
}
int32_t iAP2EANativeSession::iAP2SendPowerSourceUpdate(void)
{
    int32_t rc = IAP2_CTL_ERROR;

    iAP2PowerSourceUpdateParameter theiAP2PowerSourceUpdateParameter;

    memset(&theiAP2PowerSourceUpdateParameter, 0, sizeof(iAP2PowerSourceUpdateParameter));

    if (p_iAP2Device)
    {
        theiAP2PowerSourceUpdateParameter.iAP2AvailableCurrentForDevice = (uint16_t*)calloc(1, sizeof(uint16_t));
        if (!theiAP2PowerSourceUpdateParameter.iAP2AvailableCurrentForDevice)
        {
            LOG_INFO((tbdcl,
                    "%s() ERROR: Failed to allocate memory for iAP2AvailableCurrentForDevice",
                    __FUNCTION__));
            return IAP2_ERR_NO_MEM;
        }
        theiAP2PowerSourceUpdateParameter.iAP2DeviceBatteryShouldChargeIfPowerIsPresent = (uint8_t*)calloc(1, sizeof(uint8_t));
        if (!theiAP2PowerSourceUpdateParameter.iAP2DeviceBatteryShouldChargeIfPowerIsPresent)
        {
            LOG_INFO((tbdcl,
                    "%s() ERROR: Failed to allocate memory for iAP2DeviceBatteryShouldChargeIfPowerIsPresent",
                    __FUNCTION__));
            return IAP2_ERR_NO_MEM;
        }

        *(theiAP2PowerSourceUpdateParameter.iAP2AvailableCurrentForDevice) = 2100;
        theiAP2PowerSourceUpdateParameter.iAP2AvailableCurrentForDevice_count++;

        *(theiAP2PowerSourceUpdateParameter.iAP2DeviceBatteryShouldChargeIfPowerIsPresent) = 1;
        theiAP2PowerSourceUpdateParameter.iAP2DeviceBatteryShouldChargeIfPowerIsPresent_count++;

        rc = iAP2PowerSourceUpdate(p_iAP2Device, &theiAP2PowerSourceUpdateParameter);
        if (IAP2_OK != rc)
        {
            LOG_INFO((tbdcl,
                    "%s() ERROR: Failed iAP2PowerSourceUpdate rc=%d",
                    __FUNCTION__,rc));
        }
    }
    else
    {
        LOG_INFO((tbdcl,
                "%s() ERROR: iap2Device is NULL",
                __FUNCTION__));
    }

    free(theiAP2PowerSourceUpdateParameter.iAP2AvailableCurrentForDevice);
    theiAP2PowerSourceUpdateParameter.iAP2AvailableCurrentForDevice = NULL;
    free(theiAP2PowerSourceUpdateParameter.iAP2DeviceBatteryShouldChargeIfPowerIsPresent);
    theiAP2PowerSourceUpdateParameter.iAP2DeviceBatteryShouldChargeIfPowerIsPresent = NULL;

    return rc;
}

int32_t iAP2EANativeSession::iAP2RequestBaiduAppLaunch(bool withAlert)
{
    int32_t rc = IAP2_CTL_ERROR;
    uint8_t* iOSBundleID = NULL;
    uint8_t* iOSAppNameContainer[1] = { NULL };
    iAP2AppLaunchMethod appLaunchMethod = IAP2_LAUNCH_WITH_USER_ALERT;

    if (!withAlert)
    {
        appLaunchMethod = IAP2_LAUNCH_WITHOUT_USER_ALERT;
    }

    iAP2RequestAppLaunchParameter appLaunchParameter;

    memset(&appLaunchParameter, 0, sizeof(iAP2RequestAppLaunchParameter));
    iOSBundleID = (uint8_t*)calloc(strlen(BDCL_IAP2_BUNDLE_ID) + 1, sizeof(uint8_t)); //length+1 for '\0'

    if (!iOSBundleID)
    {
        LOG_INFO((tbdcl,
                "%s() ERROR: Failed to allocate memory for iOS app name",
                __FUNCTION__));
        return IAP2_ERR_NO_MEM;
    }

    strcpy((char*)iOSBundleID, BDCL_IAP2_BUNDLE_ID);
    iOSAppNameContainer[0] = iOSBundleID;

    if (p_iAP2Device)
    {
        //Set app name (we have only one!)
        appLaunchParameter.iAP2AppBundleID_count = 1;
        appLaunchParameter.iAP2AppBundleID = iOSAppNameContainer;
        appLaunchParameter.iAP2LaunchAlert_count = 1;
        appLaunchParameter.iAP2LaunchAlert = &appLaunchMethod;

        rc = iAP2RequestAppLaunch(p_iAP2Device, &appLaunchParameter);
        if (IAP2_OK != rc)
        {
            LOG_INFO((tbdcl,
                    "%s() ERROR: Failed to request app launch with rc=%d",
                    __FUNCTION__,rc));
        }
        else
        {
            LOG_INFO((tbdcl,
                    "%s() app='%s' requested to launch",
                    __FUNCTION__, iOSBundleID));
        }
    }
    else
    {
        LOG_INFO((tbdcl,
                "%s() ERROR: iap2Device is NULL",
                __FUNCTION__));
    }

    free(iOSBundleID);
    iOSBundleID = NULL;

    return rc;
}
void* iAP2EANativeSession::iAP2AppThread(void* exinf)
{
    int32_t rc = IAP2_CTL_ERROR;
    int32_t *pollRetVal;

    auto eaObj = static_cast<iAP2EANativeSession*>(exinf);

    prctl(PR_SET_NAME,"iap2App",0,0,0);

    //configure iap2 accessory
    eaObj->iAP2ConfigureAccessory();
    //Init iap2 device structure
    eaObj->p_iAP2Device = iAP2InitDeviceStructure(&eaObj->iap2initparam);

    LOG_INFO((tbdcl,"%s(s/n='%s') initialized device structure is %p",
            __FUNCTION__, eaObj->iAP2DevSerial.c_str(),  eaObj->p_iAP2Device));

    rc = iAP2InitDeviceConnection(eaObj->p_iAP2Device);

    LOG_INFO((tbdcl,"%s : iAP2InitDeviceConnection for device  %p rc:%d",
            __FUNCTION__,  eaObj->p_iAP2Device,rc ));

    /* create eventFd to send stop event */
    if (0 > (eaObj->mEventFd = eventfd(0, 0))) {
        LOG_ERROR((tbdcl, "create EventFd failed=%d, errno=%d", eaObj->mEventFd, errno));
        //TODO:exit with error
    }

    //start iap2PollThread here
    rc = pthread_create(&eaObj->pollThreadId, NULL, iAP2PollThread, eaObj);
    if (rc != 0)
    {
        LOG_ERROR((tbdcl, "pthread_create failed: %d %s", errno, strerror(errno)));
        //TODO:exit with error
    }

    (void)sem_wait(&(eaObj->iAP2DevReady));

    LOG_INFO((tbdcl,
            "%s(): iap2Device is Now ready .Sending powerSourceUpdate",
            __FUNCTION__));

    //todo:send this command to pollthread and let poll thread send it to device
    //device now ready . Send PowerUpdate and applaunch request
    eaObj->iAP2SendPowerSourceUpdate();

    LOG_INFO((tbdcl,
            "%s(): iap2Device is Now ready .Sending Applaunch for Baidu",
            __FUNCTION__));

    //todo:send this command to pollthread and let poll thread send it to device
    eaObj->iAP2RequestBaiduAppLaunch(false);

    //1-wait for exit command from demo app
    //2-send exit command to iAP2PollThread (implement message queue)

    //3-Wait for poll thread to exit (pthread_join)
    if (eaObj->pollThreadId > 0)
    {
        rc = pthread_join(eaObj->pollThreadId, (void**)&pollRetVal);
        if (rc != 0)
        {
            // if pthread_join fails just log the error using errno
            LOG_ERROR((tbdcl, "pthread_join failed with error %d %s", errno, strerror(errno)));
        }
        LOG_INFO((tbdcl, "iAP2 poll thread returned  and joined"));
    }

    rc = iAP2DisconnectDevice(eaObj->p_iAP2Device);
    LOG_INFO((tbdcl,"iAP2DisconnectDevice = %d", rc));

    rc = iAP2DeInitDeviceStructure(eaObj->p_iAP2Device);
    LOG_INFO((tbdcl,"iAP2DeInitDeviceStructure = %d ", rc));

    rc = iAP2DeInitializeGadget();
    LOG_INFO((tbdcl,"iAP2DeInitializeGadget = %d ", rc));

    eaObj->iAP2DeConfigureAccessory();

    //if device disconnected during active session
    //poll thread exits with rc=IAP2_DEV_NOT_CONNECTED
    //Terminate iap2 session and Baidu dession
    if (*pollRetVal == IAP2_DEV_NOT_CONNECTED )
    {
        //eaObj->mEANativeCbs.notifyEAError(eaObj->monitor, "Device Disconnected");
        //create a new event item and deligate to ServerMonitor DevEventDispatcher
        std::shared_ptr<EventItem> event(new EventItem(nullptr, EventType::EANSTOP, "", ""));
        /* queue item which played out by workqueue thread */
        eaObj->mEventDispatcher->queueEvent(event);
    }

    pthread_exit(nullptr);
}

void* iAP2EANativeSession::iAP2PollThread(void* exinf)
{
    auto eaObj =  static_cast<iAP2EANativeSession*>(exinf);
    iAP2Device_t *p_iAP2Device = (iAP2Device_t*)eaObj->p_iAP2Device;

    int32_t rc = IAP2_CTL_ERROR;

    iAP2GetPollFDs_t getPollFDs;
    iAP2PollFDs_t pollFDs[10];
    int32_t cntfds = 0;
    int32_t j = 0;
    int32_t nfds = 0; // highest-numbered file descriptor in any of the three sets, plus 1
    fd_set read_fds; // will be watched to see if characters become available for reading
    fd_set write_fds;
    struct timeval tv;
    int32_t desc_ready = 0;
    uint64_t event = 0;

    prctl(PR_SET_NAME,"iap2Poll",0,0,0);

    if (!p_iAP2Device)
    {
        LOG_INFO((tbdcl,"%s(%p) FATAL ERROR: p_iAP2Device is NULL", __FUNCTION__, exinf));
        return NULL;
    }

    LOG_INFO((tbdcl, "%s(%p) thread started", __FUNCTION__, exinf));

    //Get file descriptor
    rc = iAP2GetPollFDs(p_iAP2Device, &getPollFDs);
    for (j=0; j < getPollFDs.numberFDs; j++)
    {
        rc = eaObj->addFDToPollFDs(&pollFDs[0], j, getPollFDs.fds[j].fd, getPollFDs.fds[j].event);
        if (rc >= IAP2_OK)
        {
            cntfds = rc;
            rc = IAP2_OK;
        }
        else
        {
            LOG_INFO((tbdcl, "%s(%p) iap2AddFDToPollFDs=%d %d added",
                    __FUNCTION__, exinf, rc, getPollFDs.fds[j].event));
        }
    }

    /* add eventFd to array of file descriptors */
    pollFDs[cntfds].fd = eaObj->mEventFd;
    pollFDs[cntfds].event = POLLIN;
    cntfds++;

    //Main loop
    while (false == eaObj->endPoll)
    {
        /* Wait up to five seconds if no fd becomes ready. */
         tv.tv_sec = 1;
         tv.tv_usec = 0;

        /* FD_ZERO() clears out the fd_set, so it doesn't contain any file descriptors */
        FD_ZERO(&read_fds);
        FD_ZERO(&write_fds);

        rc = eaObj->addFDsToFDset(&pollFDs[0], cntfds, &nfds, &read_fds, &write_fds);
        if (rc != IAP2_OK)
        {
            LOG_INFO((tbdcl, "%s(%p) ERROR: Failed to call iap2AddFDsToFDset with rc=%d",
                    __FUNCTION__, exinf, rc));
        }

        //rc = select(nfds+1, &read_fds, &write_fds, NULL, NULL);
        desc_ready = select(nfds+1, &read_fds, &write_fds, NULL, &tv);
        if (desc_ready > 0)
        {
            for (j =0 ; (j < cntfds) && (desc_ready > 0); j++)
            {
                if ((j < cntfds) && (FD_ISSET(pollFDs[j].fd, &read_fds)))
                {
                    /* eventfd triggered at file descriptor mEventFd */
                    if (pollFDs[j].fd == eaObj->mEventFd)
                    {
                        /* read eventfd to get current number of occurred events */
                        if (eventfd_read(eaObj->mEventFd, &event) != 0) {
                            LOG_ERROR((tbdcl, "%s(%p) eventfd_read(fd=%d) failed",
                                    __FUNCTION__, exinf, eaObj->mEventFd));
                        } else {
                            LOG_INFO((tbdcl, "%s(%p) eventfd triggered to stop iAP2PollThread",
                                    __FUNCTION__, exinf));
                        }
                    }
                    else
                    {
                        LOG_INFO((tbdcl, "%s(%p) read fd[%d] %d (event: %d) is set",
                                __FUNCTION__, exinf, j, pollFDs[j].fd, pollFDs[j].event));

                        rc = iAP2HandleEvent(p_iAP2Device, pollFDs[j].fd, pollFDs[j].event);
                    }

                    desc_ready--;
                }
                if( (j < cntfds) && (FD_ISSET(pollFDs[j].fd, &write_fds)) )
                {
                    LOG_INFO((tbdcl, "%s(%p) write fd[%d] %d (event: %d) is set",
                            __FUNCTION__, exinf, j, pollFDs[j].fd, pollFDs[j].event));

                    rc = iAP2HandleEvent(p_iAP2Device, pollFDs[j].fd, pollFDs[j].event);

                    desc_ready--;
                }

                if (IAP2_DEV_NOT_CONNECTED == rc)
                {
                    LOG_INFO((tbdcl, "%s(%p) ERROR: device disconnected", __FUNCTION__, exinf));
                    eaObj->endPoll = true;

                    break;
                }
            }
        }
        else
        {
            /* rc = 0 : select() timed out
             * rc < 0 : indicates an error (check errno)
             */

            if (desc_ready < 0)
            {
                LOG_WARN((tbdcl, "%s(%p)  myPollFDs failed = %d (errno='%s')", __FUNCTION__, exinf, desc_ready, strerror(errno)));
                eaObj->endPoll = true;
            }
            else
            {
                LOGD_DEBUG((tbdcl, "%s(%p)  myPollFDs timed out", __FUNCTION__, exinf));
            }
            rc = desc_ready;
        }
    }

    LOG_INFO((tbdcl, "%s(%p) exit", __FUNCTION__, exinf));
    pthread_exit((void*)&rc);
}


} }
